/*
 * mp_mod.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong <swong@ieee.org>
 *
 * Dec 14, 2000 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

// Set r to a * b (mod m).
//
// Classical modular multiplication
// (Algorithm 14.28 on p.600 of HAC)
//
// Note : - For speed, this function would not check the validity of
//          the inputs, i.e. a, b and m are positive integers
//        - Again for speed, the inputs a and b should be < m.
Int16 MP_mod_mul(INT *r, INT *a, INT *b, INT *m, MP_CTX *ctx)
{
INT *t;
Int16 ret=0;

	t = ctx->t[ctx->tos++];

	if (a == b) { if (!MP_sqr(t,a)) goto err; }
	else { if (!MP_mul(t,a,b)) goto err; }

	if (!MP_div(NULL, r, t, m, ctx)) goto err;
	ret=1;
err:
	ctx->tos--;
	return(ret);
}


// Set r to a^b mod m.
//
// Classical Left-to-right binary exponentiation (repeated square-and-multiply)
// (Algorithm 14.79 on p.615 of HAC)
//
// Note : r can be a.
Int16 MP_mod_exp(INT *r, INT *a, INT *b, INT *m, MP_CTX *ctx)
{
Int16 i, bits, ret=0;
INT *A, *t1, *t2;

	t1 = ctx->t[ctx->tos++];
	t2 = ctx->t[ctx->tos++];
	A = ctx->t[ctx->tos++];

  i = MP_cmp(a, m);
  if(!i) { MP_zero(r); ret=1; goto err; }
  else if(i > 0) { if(!MP_div(NULL, t2, a, m, ctx)) goto err; }
  else { if(MP_copy(t2, a) == NULL) goto err; }

	bits = MP_num_bits(b) - 1;
  if(MP_copy(A, t2) == NULL) goto err;

  for (i=--bits; i>=0; i--) {
    if(!MP_sqr(t1, A)) goto err;
    if(!MP_div(NULL, A, t1, m, ctx)) goto err;
    if(MP_is_bit_set(b, i)) {
      if(!MP_mul(t1, A, t2)) goto err;
      if(!MP_div(NULL, A, t1, m, ctx)) goto err;
    }
  }

  if(MP_copy(r, A) == NULL) goto err;
	ret=1;
err:
	ctx->tos -= 3;
	return(ret);
}
