/*
 * mp_lib.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong  <swong@ieee.org>
 *
 * Dec 14, 2000 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

// Initialize and return a memory chunk of size MP_DEFAULT_BITS bits.
// Each byte of the memory chunk is also initialized to zero. Each
// INT variable should normally be initialized before use, and
// freed using MPLibMP_free between each initialization. Use
// MPLibMP_clear_free instead if the variable contains some
// cryptographic secret. It clears out the memory chunk by writing
// zero to each byte and then deallocates the memory chunk.
//
// Example:
// {
//   INT *integ;
//   integ = MPLibMP_new(MPLibRefNum);
//   ...
//   MPLibMP_add(MPLibRefNum, integ, ...);
//   ...
//   MPLibMP_sub(MPLibRefNum, integ, ...);
//   ...
//   MPLibMP_clear_free(MPLibRefNum, integ);
// }
INT *MP_new()
{
INT *ret;
DIGIT *p;

	ret=(INT *) MemPtrNew(sizeof(INT));
	if (ret == NULL) {
    ErrDisplay("Memory allocation error in MP_new. (1)");
    return(NULL);
  }
	ret->top=0;
	ret->neg=0;
	ret->max=(MP_DEFAULT_BITS/DIGIT_BITS);
	p=(DIGIT *) MemPtrNew(sizeof(DIGIT)*(ret->max));
	if (p == NULL) {
    ErrDisplay("Memory allocation error in MP_new. (2)");
    return(NULL);
  }
	ret->d=p;

	memset(p,0,(ret->max)*sizeof(DIGIT));
	return(ret);
}


// Clear the content of integer by setting each byte of integer to zero
// but not freeing the memory of integer
void MP_clear(INT *integer)
{
	memset(integer->d, 0, integer->max*sizeof(integer->d[0]));
	integer->top = 0;
	integer->neg = 0;
}


// Free the memory of integer but would not clear the content.
// It is fine to use this function for all (INT *) variables
// when you are doen with them. But MP_clear_free is recommended
// for cryptographic applications.
void MP_free(INT *integer)
{
	if (integer == NULL) return;
	if (integer->d != NULL) MemPtrFree(integer->d);
	MemPtrFree(integer);
}


// Clear the content of integer by setting each byte of integer to zero
// and then free the memory of the variable.
void MP_clear_free(INT *integer)
{
	if (integer == NULL) return;
	if (integer->d != NULL) {
		memset(integer->d, 0, integer->max*sizeof(integer->d[0]));
		MemPtrFree(integer->d);
  }
	memset(integer, 0, sizeof(INT));
	MemPtrFree(integer);
}


// Initialize the temporary buffer/variable
MP_CTX *MP_CTX_new()
{
MP_CTX *ret;
INT *n;
Int16 i,j;

	ret=(MP_CTX *) MemPtrNew(sizeof(MP_CTX));
	if (ret == NULL) {
    ErrDisplay("Memory allocation error in MP_CTX_new (1)");
    return(NULL);
  }

	for (i=0; i<MP_CTX_NUM; i++) {
		n = MP_new();
		if (n == NULL) {
      for (j=0; j<i; j++) MP_free(ret->t[j]);
      MemPtrFree(ret);
    }
		ret->t[i]=n;
  }

	// There is actually an extra one, this is for debugging
	ret->t[MP_CTX_NUM]=NULL;

	ret->tos=0;
	return(ret);
}


// Clear and free the temporary buffer
void MP_CTX_free(MP_CTX *c)
{
Int16 i;

  if (c == NULL) return;
	for (i=0; i<MP_CTX_NUM; i++)
		MP_clear_free(c->t[i]);
	MemPtrFree(c);
}


// Set a to the n low-bit of a
//
// Note : |a| <= n
Int16 MP_mask_bits(INT *a, Int16 n)
{
Int16 b,w;

	w = n/DIGIT_BITS;
	b = n%DIGIT_BITS;

	if (w >= a->top) return(0);

	if (b == 0) a->top=w;
	else {
		a->top=w+1;
		a->d[w]&= ~(DIGIT_MASK<<b);
		while ((w >= 0) && (a->d[w] == 0)) {
			a->top--;
			w--;
    }
  }
	return(1);
}


// Copy from (INT *)src to (INT *)dest
//
// Return the pointer of (INT *)dest
//
// Note : Support overlapping buffers
INT *MP_copy(INT *dest, INT *src)
{
	if (MP_alloc(dest, src->top*DIGIT_BITS) == NULL) return(NULL);
	memcpy(dest->d, src->d, sizeof(DIGIT)*src->top);
	dest->top=src->top;
	dest->neg=src->neg;
	return(dest);
}


void *memset(void *b, UInt8 c, Int32 len)
{
	MemSet(b, len, c);
	return b;
}


// Support overlapping buffers but slow
// Don't use it if dst and src are not overlapped
void *memcpy(void *dst, void *src, Int32 len)
{
int i;
unsigned char *tmp, *P1, *P2;

	if(len) {
		tmp = (unsigned char *) MemPtrNew(len);
		if(tmp == NULL) {
			ErrDisplay("Memory allocation error in memcpy");
			return(NULL);
		}
		P1 = (unsigned char *) tmp;
		P2 = (unsigned char *) src;

		for(i=0; i<len; i++) {
			*P1 = *P2;
			P1++;
			P2++;
		}

		P1 = (unsigned char *) dst;
		P2 = tmp;
		for(i=0; i<len; i++) {
			*P1 = *P2;
			P1++;
			P2++;
		}

		MemPtrFree(tmp);

		return(dst);
	}

	return(NULL);
}


// PalmOS doesn't provide a good equivalent to realloc, so we roll our own
void *PalmOS_realloc(void *ptr, UInt32 size)
{
Err err;
UInt32 oldsize, copysize;
void *newptr;

  // Remember the special cases
	if (!size) {
		MemPtrFree(ptr);
		return NULL;
	}

  if (!ptr) return MemPtrNew(size);

  // First try a simple Resize.  This will only work if the ptr doesn't have to move
  err = MemPtrResize(ptr, size);
  if (!err) return ptr;

  // Darn.  Let's play a game.
  oldsize = MemPtrSize(ptr);
  copysize = (oldsize < size) ? oldsize : size;
  newptr = MemPtrNew(size);
  if (!newptr) return newptr;  // Note that ptr is still valid

  MemMove(newptr, ptr, copysize);
  MemPtrFree(ptr);
  return newptr;
}


// Allocate enough memory chunk for b to store a bits/DIGIT_BITS digits
// of integer.
//
// Return a pointer to b if succeed; otherwise return NULL.
INT *MP_expand(INT *b, UInt32 bits)
{
register UInt32 n;
DIGIT *p;

	while (bits > b->max*DIGIT_BITS) {
		//n=((bits+DIGIT_BITS-1)/DIGIT_BITS)*2;  // what times 2?
		n=((bits+DIGIT_BITS-1)/DIGIT_BITS);
		//p=b->d=(DIGIT *)Realloc(b->d,sizeof(DIGIT)*(n+1));
		p=b->d=(DIGIT *)Realloc(b->d,sizeof(DIGIT)*n);
		if (p == NULL) {
      ErrDisplay("Memory allocation error in MP_expand.");
			return(NULL);
    }
		memset(&(p[b->max]),0,((n+1)-b->max)*sizeof(DIGIT));
		b->max=n;
  }

	return(b);
}
