/*
 * mp_gcd.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong <swong@ieee.org>
 *
 * Dec 14, 2000 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

static Int16 binary_gcd(INT *a, INT *b);


// Set d to gcd(a, b).
//
// Classical Euclidean Algorithm (Algorithm 2.104 on p. 66 of HAC).
//
// Note : - d can be a or b.
//        - According to my implementation results, the Binary gcd algorithm
//          (Algorithm 14.54) is SLOWER than the classical Euclidean algorithm
//          (Algorithm 2.104) in most of the cases.
Int16 MP_gcd(INT *d, INT *a, INT *b, MP_CTX *ctx)
{
INT *t1, *t2, *t3, *t4;
Int16 ret=0;

	t1=ctx->t[ctx->tos];
	t2=ctx->t[ctx->tos+1];
  ctx->tos+=2;

	if (MP_copy(t1, a) == NULL) goto err;
	if (MP_copy(t2, b) == NULL) goto err;
  t3 = d;
  while(!MP_is_zero(t2)) {
    if (MP_div(NULL, t3, t1, t2, ctx) == 0) goto err;
    t4 = t1;
    t1 = t2;
    t2 = t3;
    t3 = t4;
  }
  if(MP_copy(d, t1) == NULL) return(0);
  ret = 1;
err:
	ctx->tos-=2;
  return(ret);
}


// Set d to gcd(a, b).
//
// Binary gcd Algorithm (Algorithm 14.54 on p. 606 of HAC)
//
// Note : - d can be a or b.
//        - According to my implementation results, the Binary gcd algorithm
//          (Algorithm 14.54) is SLOWER than the classical Euclidean algorithm
//          (Algorithm 2.104) in most of the cases.
Int16 MP_binary_gcd(INT *d, INT *a, INT *b, MP_CTX *ctx)
{
INT *t1,*t2,*tmp;

	t1=ctx->t[ctx->tos];
	t2=ctx->t[ctx->tos+1];

	if (MP_copy(t1,a) == NULL) return(0);
	if (MP_copy(t2,b) == NULL) return(0);

	if (MP_cmp(t1,t2) < 0) { tmp=t1; t1=t2; t2=tmp; }
	if (!binary_gcd(t1, t2)) return(0);

	if (MP_copy(d,t1) == NULL) return(0);
	return(1);
}


// Binary gcd algorithm (Algorithm 14.54 on p.606 of HAC)
//
// Set a to gcd(a, b)
//
// Note : We assume initially a > b
static Int16 binary_gcd(INT *a, INT *b)
{
INT *t1, *t2, *tmp;
Int16 shifts=0;

  t1 = a;
  t2 = b;
	for (;;) {
		if (MP_is_zero(t2)) break;

		if (MP_is_odd(t1)) {
			if (MP_is_odd(t2)) {
				if (!MP_sub(t1,t1,t2)) return(0);
				if (!MP_rshift1(t1,t1)) return(0);
				if (MP_cmp(t1,t2) < 0) {
          // use t1 and t2 instead of a and b directly 'cause the pointer a
          // and b are constant, you cannot change the value of a and b by
          // something like a = b and b = t.
          tmp=t1; t1=t2; t2=tmp;
        }
      } else {  // t1 odd - t2 even
				if (!MP_rshift1(t2,t2)) return(0);
				if (MP_cmp(t1,t2) < 0) { tmp=t1; t1=t2; t2=tmp; }
      }
    } else {  // t1 is even
			if (MP_is_odd(t2)) {
				if (!MP_rshift1(t1,t1)) return(0);
				if (MP_cmp(t1,t2) < 0) { tmp=t1; t1=t2; t2=tmp; }
      } else {  // t1 even - t2 even
				if (!MP_rshift1(t1,t1)) return(0);
				if (!MP_rshift1(t2,t2)) return(0);
				shifts++;
      }
    }
  }

	if (shifts) if (!MP_lshift(t1,t2,shifts)) return(0);

  // copy back to a if (t1 != a)
  if (a != t1) if(MP_copy(a, t1) == NULL) return(0);
	return(1);
}


// Extended Euclidean algorithm (Algorithm 2.107 on p.67 of HAC)
//
// Return x where ax == 1 (mod n).
//
// Note: - a can be > n
//       - return NULL if gcd(a, n) > 1
INT *MP_mod_inverse(INT *a, INT *n, MP_CTX *ctx)
{
INT *A,*B,*X,*Y,*M,*D,*R;
INT *ret=NULL,*T;
Int16 sign;

	A=ctx->t[ctx->tos];
	B=ctx->t[ctx->tos+1];
	X=ctx->t[ctx->tos+2];
	D=ctx->t[ctx->tos+3];
	M=ctx->t[ctx->tos+4];
	Y=ctx->t[ctx->tos+5];
	ctx->tos+=6;
	R = MP_new();
	if (R == NULL) goto err;

	MP_zero(X);
	MP_one(Y);
	if (MP_copy(A,a) == NULL) goto err;
	if (MP_copy(B,n) == NULL) goto err;
	sign=1;

	while (!MP_is_zero(B)) {
		if (!MP_div(D,M,A,B,ctx)) goto err;
		T=A;
		A=B;
		B=M;

		if (!MP_mul(T,D,X)) goto err;
		if (!MP_add(T,T,Y)) goto err;
		M=Y;
		Y=X;
		X=T;
		sign= -sign;
  }
	if (sign < 0) if (!MP_sub(Y,n,Y)) goto err;

	if (MP_is_one(A)) {
    if (!MP_div(NULL,R,Y,n,ctx)) goto err;
  } else {
		goto err;
  }
	ret=R;
err:
	if ((ret == NULL) && (R != NULL)) MP_free(R);
	ctx->tos-=6;
	return(ret);
}
