/*
 * mp_div.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong <swong@ieee.org>
 *
 * Dec 14, 2000 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

// Set q and r to the quotient and remainder of (dividend / divisor).
//
// Note : - For computing modular reduction (i.e. dividend mod divisor),
//          call the function as MP_div(NULL, r, dividend, divisor, ctx).
//        - Check 14.2.5 on p.598 of HAC for details.
//
// Division: (dividend) divided by (divisor) equals (quotient) plus (remainder).
Int16 MP_div(INT *q, INT *r, INT *dividend, INT *divisor, MP_CTX *ctx)
{
Int16 norm_shift, i, j, loop;
INT *tmp, wdividend, *sdividend, *sdivisor, *result;
DIGIT *result_p,*wdividend_p;
DIGIT d0, d1;
Int16 dividend_n, divisor_n;

	if (MP_is_zero(dividend)) {
    if (q != NULL) { q->top = 0; q->d[0] = 0; }
    return(1);
  }
	if (MP_is_zero(divisor)) {
		return(0);
  }
	if (MP_ucmp(dividend,divisor) < 0) {
		if (r != NULL) if (MP_copy(r,dividend) == NULL) return(0);
		if (q != NULL) { q->top=0; q->d[0] = 0; }
		return(1);
  }

	tmp=ctx->t[ctx->tos]; 
	tmp->neg=0;
	sdividend = ctx->t[ctx->tos+1];
	sdivisor  = ctx->t[ctx->tos+2];
	if (q == NULL)
		result = ctx->t[ctx->tos+3];
	else result = q;

	// First we normalise the numbers (check 14.23 on p.598 of HAC for details)
	norm_shift = DIGIT_BITS - ((MP_num_bits(divisor)) % DIGIT_BITS);
	MP_lshift(sdivisor, divisor, norm_shift);
	sdivisor->neg=0;
	norm_shift += DIGIT_BITS;
	MP_lshift(sdividend, dividend, norm_shift);
	sdividend->neg=0;
	divisor_n  = sdivisor->top;
	dividend_n = sdividend->top;
	loop = dividend_n - divisor_n;

	// setup a 'window' into sdividend
	// This is the part that corresponds to the current 'area' being divided
	wdividend.d   = &(sdividend->d[loop]);
	wdividend.top = divisor_n;
	wdividend.max = sdividend->max; // a bit of a lie
	wdividend.neg = 0;

	// Get the top 2 words of sdivisor
	d0=sdivisor->d[divisor_n-1];
	d1=(divisor_n == 1)?0:sdivisor->d[divisor_n-2];

	// pointer to the 'top' of snum
	wdividend_p= &(sdividend->d[dividend_n-1]);

	// Setup to 'result'
	result->neg = (dividend->neg^divisor->neg);
	result->top = loop;
	if (!MP_alloc(result,(loop+1)*DIGIT_BITS)) return(0);
	result_p= &(result->d[loop-1]);

	// space for temp
	if (!MP_alloc(tmp,(divisor_n+1)*DIGIT_BITS)) return(0);

	if (MP_ucmp(&wdividend,sdivisor) >= 0) {
		MP_qsub(&wdividend,&wdividend,sdivisor);
		*result_p=1;
		result->d[result->top-1]=1;
  } else
		result->top--;
	result_p--;

	for (i=0; i<loop-1; i++) {
		DIGIT qu,n0,n1;
		DIGIT l0;

		wdividend.d--; wdividend.top++;
		n0=wdividend_p[0];
		n1=wdividend_p[-1];
		if (n0 == d0)
			qu=DIGIT_MASK;
		else
			qu=MP_div2(n0,n1,d0);
		{
		DIGIT t1l,t1h,t2l,t2h,t3l,t3h,ql,qh,t3t;
		t1h=n0;
		t1l=n1;
		for (;;) {
			t2l=LBITS(d1); t2h=HBITS(d1);
			ql =LBITS(qu);  qh =HBITS(qu);
			mul64(t2l,t2h,ql,qh);

			t3t=LBITS(d0); t3h=HBITS(d0);
			mul64(t3t,t3h,ql,qh);
			t3l=(t1l-t3t);
			if (t3l > t1l) t3h++;
			t3h=(t1h-t3h);

			if (t3h) break;
			if (t2h < t3l) break;
			if ((t2h == t3l) && (t2l <= wdividend_p[-2])) break;

			qu--;
    }
		}
		l0=MP_mul_digit(tmp->d,sdivisor->d,divisor_n,qu);
		tmp->d[divisor_n]=l0;
		for (j=divisor_n+1; j>0; j--)
			if (tmp->d[j-1]) break;
		tmp->top=j;

		j=wdividend.top;
		MP_sub(&wdividend,&wdividend,tmp);

		sdividend->top=sdividend->top+wdividend.top-j;

		if (wdividend.neg) {
			qu--;
			j=wdividend.top;
			MP_add(&wdividend,&wdividend,sdivisor);
			sdividend->top+=wdividend.top-j;
    }
		*(result_p--)=qu;
		wdividend_p--;
  }
	if (r != NULL) {
		MP_rshift(r, sdividend, norm_shift);
		r->neg = dividend->neg;
  }
	return(1);
}


// Set return to (h || l)/d
DIGIT MP_div2(DIGIT h, DIGIT l, DIGIT d)
{
	return((DIGIT)(((((DOUBLE_DIGIT)h)<<DIGIT_BITS)|l)/(DOUBLE_DIGIT)d));
}
