/*
 * mp_crt.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong <swong@ieee.org>
 *
 * Feb 11, 2001 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

// Find the unique solution 0 <= x < p*q given that x \equiv v1 \pmod{p} and x \equiv v2 \pmod{q}.
// (i.e. x = (((v2 - v1) * u) mod q) * p + v1 where u = p^{-1} mod q and x < n)
//
// Garner's algorithm for CRT (Chinese Remainder Theorem)
// for special case of two moduli
//                     ^^^^^^^^^^
// e.g. particularly efficient for RSA moduli n = pq
//
// Note: - This function would first check if p and q are relatively prime
//       - This function assumes that v1 < p and v2 < q.
//       - Return 1 is succeed; otherwise, 0.
//       - Check HAC p.612 Algorithm 14.71 and Note 14.75(i) for details
Int16 MP_crt2(INT *x, INT *v1, INT *v2, INT *p, INT *q, MP_CTX *ctx)
{
INT *u=NULL;
INT *d, *t1, *t2;
Int16 ret=0;

	d = ctx->t[ctx->tos];
	t1 = ctx->t[ctx->tos+1];
	t2 = ctx->t[ctx->tos+2];
	ctx->tos+=3;

	// check if p and q are relatively prime
	if(!MP_gcd(d, p, q, ctx)) goto err;
	if(!MP_is_one(d)) goto err;

	u = MP_mod_inverse(p, q, ctx);
	if(u == NULL) goto err;

	if(!MP_sub(t1, v2, v1)) goto err;
	if(MP_cmp(v2, v1) < 0)
		if(!MP_add(t1, t1, q)) goto err;
	if(!MP_mod_mul(t2, t1, u, q, ctx)) goto err;
	if(!MP_mul(t1, t2, p)) goto err;
	if(!MP_add(x, t1, v1)) goto err;

	ret = 1;
err:
	ctx->tos-=3;
	if(u != NULL) MP_free(u);
  return(ret);
}