/*
 * mp_conv.c
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong  <swong@ieee.org>
 *
 * Dec 15, 2000 - Initial Version
 */
#include <PalmOS.h>
#include "mp.h"
#include "mp_priv.h"

// Convert a binary stream s to INT and return it as (INT *).
// len is the length of s in number of bytes.
//
// Note: ignore negative (s->neg)
INT *MP_bin2bn(UInt8 *s, UInt16 len)
{
INT *ret = NULL;
UInt16 n, i, m;
DIGIT l;

	ret = MP_new();
	if (ret == NULL) {
    ErrDisplay("Memory allocation error in MP_bin2bn");
    return(NULL);
  }

	l=0;
	n=len;
	if (n == 0) {
		ret->top=0;
		return(ret);
  }
	//if (MP_alloc(ret, (UInt32) (n+2)*DIGIT_BITS) == NULL)
	if (MP_alloc(ret, (UInt32) (n*DIGIT_BITS)) == NULL)
		return(NULL);

	i=((n-1)/DIGIT_BYTES)+1;
	m=((n-1)%(DIGIT_BYTES));
	ret->top=i;
	while (n-- > 0) {
		l=(l<<8L)| *(s++);
		if (m-- == 0) {
			ret->d[--i]=l;
			l=0;
			m=DIGIT_BYTES-1;
    }
  }
	// need to call this due to clear byte at top if avoiding
	// having the top bit set (-ve number)
	MP_fix_top(ret);
	return(ret);
}


// Convert (INT *) a to a binary stream (UInt8 *) to.
//
// Return length of (UInt8 *) to in number of bytes.
//
// Note: ignore negative
UInt16 MP_bn2bin(INT *a, UInt8 *to)
{
UInt16 n,i;
DIGIT l;
UInt8 *tmpP;

	tmpP = to;
	n = i = MP_num_bytes(a);
	while (i-- > 0) {
		l = a->d[i/DIGIT_BYTES];

		*(tmpP++)=(UInt8)(l>>(8*(i%DIGIT_BYTES)))&0xff;
  }

	return(n);
}


// Convert a number in ASCII (Char *) a to INT and return it as (INT *).
// a is represented in HEX.
INT *MP_ascii2bn(Char *a)
{
INT *ret=NULL;
DIGIT l=0, c, k;
Int16 neg = 0;
Int16 i, j, m, h;

	if ((a == NULL) || (*a == '\0')) return(NULL);
	if (*a == '-') { neg=1; a++; }

  ret = MP_new();

	for (i=0; (a[i]>='0' && a[i]<='9') || (a[i]>='a' && a[i]<='f')
		|| (a[i]>='A' && a[i]<='F'); i++) ;

	if (MP_alloc(ret,i*4) == NULL) {
    MP_free(ret);
    return(NULL);
  }

	j=i; // pointing to the least significant nibble
	m=0;
	h=0;
	while (j > 0) {
		m=((DIGIT_BYTES*2) <= j)?(DIGIT_BYTES*2):j;
		l=0;
		for (;;) {
			c=a[j-m];
			if ((c >= '0') && (c <= '9')) k=c-'0';
			else if ((c >= 'a') && (c <= 'f')) k=c-'a'+10;
			else if ((c >= 'A') && (c <= 'F')) k=c-'A'+10;
			else k=0; // paranoia
			l=(l<<4)|k;

			if (--m <= 0) {
				ret->d[h++]=l;
				break;
      }
    }
		j-=(DIGIT_BYTES*2);
  }
	ret->top=h;
	MP_fix_top(ret);
	ret->neg=neg;

	return(ret);
}


// Convert (INT *) a to a hex string in ASCII and return it as (Char *).
Char *MP_bn2ascii(INT *a)
{
Char *hs="0123456789ABCDEF";
Char *strP, *tmpP;
Int16 i, j, z=0;
UInt8 v;

	if(a == NULL) return(NULL);
	
	strP=(Char *) MemPtrNew(a->top*DIGIT_BYTES*2+2);
	if (strP == NULL) {
    ErrDisplay("Memory allocation error in MP_bn2ascii");
    return(NULL);
  }

	tmpP=strP;
	if (a->neg) *(tmpP++)='-';
	if (a->top == 0) *(tmpP++)='0';
	for (i=a->top-1; i >=0; i--) {
		for (j=DIGIT_BITS-8; j >= 0; j-=8) {
			// strip leading zeros
			v=((UInt8)(a->d[i]>>(long)j))&0xff;
			if (z || (v != 0)) {
				*(tmpP++) = hs[v>>4];
				*(tmpP++) = hs[v&0x0f];
				z=1;
      }
    }
  }
	*tmpP='\0';

  return(strP);
}


// Return the number of bits of (INT *) a.
UInt32 MP_num_bits(INT *a)
{
DIGIT l;
UInt32 i;

	if (a->top == 0) return(0);
	l=a->d[a->top-1];
	i=(a->top-1)*DIGIT_BITS;
	if (l == 0)
		ErrDisplay("Bad TOP value in MP_num_bits");

	return(i+MP_num_bits_digit(l));
}


// Return number of bits of a digit (DIGIT) l
UInt16 MP_num_bits_digit(DIGIT l)
{
char bits[256]={
	0,1,2,2,3,3,3,3,4,4,4,4,4,4,4,4,
	5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
	8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
};

#if defined(THIRTY_TWO_BIT)
  if (l & 0xffff0000L) {
    if (l & 0xff000000L) return(bits[l>>24L]+24);
    else return(bits[l>>16L]+16);
  }
#endif
#if defined(SIXTEEN_BIT) || defined(THIRTY_TWO_BIT)
  if (l & 0xff00L) return(bits[l>>8]+8);
#endif
  return(bits[l]);
}
