/*
 * MPLib.c - multiple-precision integer arithmetic shared library
 *
 * This code is in the public domain. I would appreciate bug reports and
 * enhancements.
 *
 * Duncan S Wong  <swong@ieee.org>
 *
 * Feb 11, 2001 - v0.2
 * Dec 14, 2000 - Initial Version
 */

// Because we play with #defines that you're not normally expected to play with,
// we tend to run into cryptic link errors by including precompiled headers:
#ifndef PILOT_PRECOMPILED_HEADERS_OFF
	#define	PILOT_PRECOMPILED_HEADERS_OFF
#endif

#include <PalmOS.h>			// for OS 3.5, previously Pilot.h
#include "MPLib.h"			// Our interface definition
#include "MPLibPrv.h"		// Private routiens (globals stuff)

#pragma mark - 
// Utility functions for globals structure access

// *****
// * FUNCTION: 		AllocGlobals
// * 
// * DESCRIPTION:	Allocate AND LOCK library globals ptr for a given RefNum
// *
// * PARAMETERS:	uRefNum		-		Lib refnum whose globals we'll create
// *			
// * RETURNED:		Ptr to new globals  - success
// *		      		NULL                - failure, MemHandleNew failed (!)
// *
// * POSTCONDITION:	Since this routine locks the handle it returns if successful, the
// *				caller is responsible for calling MSLUnlockGlobals() when s/he is done.
// *				Remember, the sooner you do this, the more you prevent heap fragmentation.
// *
// *	JeffI	05/18/99	Initial Implementation
// *****
GlobalsTypePtr AllocGlobals(UInt16 uRefNum)
{
SysLibTblEntryPtr  sysLibEntryP;
GlobalsTypePtr     gP = NULL;
MemHandle          gH = NULL;
	
	ErrFatalDisplayIf(sysInvalidRefNum == uRefNum, "Invalid refnum.");

	// Fetch a ptr to our lib's table entry in the OS's array (refnum is index)
	sysLibEntryP = SysLibTblEntry(uRefNum);
	ErrFatalDisplayIf(NULL == sysLibEntryP, "Invalid refnum.");
	ErrFatalDisplayIf(sysLibEntryP->globalsP, "Lib globals ptr already exists.");
	
	gH = MemHandleNew(sizeof(GlobalsType));				// Alloc mem for globals here
	if ( !gH ) return ( NULL );
		
	sysLibEntryP->globalsP = (void*)gH;							// Store handle in lib entry
	
	gP = (GlobalsTypePtr) LockGlobals(uRefNum);			
	ErrFatalDisplayIf(!gP, "Unable to lock lib globals ptr.");

	// We need to set the owner of this chunk to 'system'.  If we don't do this, then
	// the memory manager will automatically free this when the first application to
	// call Open() exits.  Since we intend this library (and its globals) to hang around
	// regardless of which app begins and exits, we need to do this:
	MemPtrSetOwner(gP, 0);									// 0 == OS
	
	MemSet(gP, sizeof(GlobalsType), 0);  // Clean it out

	// Globals should be initialized in your lib's Open entry point... see Open()
	return ( gP );
}


// *****
// * FUNCTION: 		FreeGlobals
// * 
// * DESCRIPTION:	Deallocate a lib's globals ptr, given its RefNum.
// *
// * PARAMETERS:	uRefNum	-	Lib refnum whose globals we'll deallocate
// *			
// * RETURNED:		ErrNone	-	MemHandleNew success
// *	      			!0			-	failure, MemHandleNew failed 
// *
// *	JeffI	05/18/99	Initial Implementation
// *****
Err FreeGlobals(UInt16 uRefNum)
{
SysLibTblEntryPtr sysLibEntryP;
MemHandle   			gH = NULL;
	
	ErrFatalDisplayIf(sysInvalidRefNum == uRefNum, "Invalid refnum.");
	
	sysLibEntryP = SysLibTblEntry(uRefNum );
	ErrFatalDisplayIf(NULL == sysLibEntryP, "Invalid refnum.");
	
	gH = (MemHandle) (sysLibEntryP->globalsP);	// Get our globals handle
	ErrFatalDisplayIf(!gH, "Lib globals ptr does not exist.");

	sysLibEntryP->globalsP = NULL;
	return( MemHandleFree(gH) );
}

// *****
// * FUNCTION: 		LockGlobals
// * 
// * DESCRIPTION:	Return a ptr to a particular lib's MSLGlobalsType structure
// *
// * PARAMETERS:	uRefNum		-	Lib refnum whose globals we'll lock
// *			
// * RETURNED:		0		    	-	Caller needs to allocate them first with MSLAllocGlobals()!
// *			      	Valid ptr	-	success
// *
// * POSTCONDITION:	If I return 0, the caller needs to MSLAllocGlobals().
// *
// *	JeffI	05/18/99	Initial Implementation
// *****
GlobalsTypePtr LockGlobals(UInt16 uRefNum)
{
GlobalsTypePtr  		gP 				= NULL;						// Necessary!
SysLibTblEntryPtr		sysLibEntryP 	= NULL;
MemHandle	      		gH				= NULL;
	
	ErrFatalDisplayIf(sysInvalidRefNum == uRefNum, "Invalid refnum.");
	
	sysLibEntryP = SysLibTblEntry(uRefNum);
	ErrFatalDisplayIf(NULL == sysLibEntryP, "Invalid refnum.");
	
	gH = (MemHandle) (sysLibEntryP->globalsP);

	// We don't ErrFatalDisplay here if !gH.  This is so the caller can check the return
	// value and if it's null, the caller knows s/he needs to MSLAllocGlobals(), similar
	// to the behavior of SysLibFind() and SysLibLoad()ing something.
	if (gH) gP = (GlobalsTypePtr) MemHandleLock(gH);
	
	// Notice we want to return NULL if this handle hasn't yet been allocated!
	return gP;
}

// *****
// * FUNCTION: 		UnlockGlobals
// * 
// * DESCRIPTION:	Unlock a ptr to a MSLGlobalsType structure
// *
// * PRECONDITION:	gP has been locked down by a call to MSLLoclGlobals.
// *
// * PARAMETERS:	gP		  	-	Locked ptr to structure
// *			
// * RETURNED:		!0	  		-	MemPtrUnlock failure (!)
// *		      		ErrNone		-	MemPtrUnlock success
// *
// *	JeffI	05/18/99	Initial Implementation
// *****
Err UnlockGlobals(GlobalsTypePtr gP)
{
	return(MemPtrUnlock(gP));											// No magic here..
}



#pragma mark -
// OS-Required entry point implementations:

// *****
// * FUNCTION: 		Open
// * 
// * DESCRIPTION:	Open the library and alloc globals if necessary
// *
// * PRECONDITION:	Caller has already done a SysLibFind and SysLibLoad to get refnum
// *
// * PARAMETERS:	uRefNum		-		Lib refnum 
// *			
// * RETURNED:		MPErrNone		  	-	success
// *			      	MPErrNoGlobals	- unable to allocate globals
// *
// * POSTCONDITION:	Caller should Close() this lib as soon as s/he is done using it.
// *				Multiple Opens() are ok, but each one should always have a 
// *				correspoding Close() to balance it out.
// *
// *	JeffI	05/19/99	Initial Implementation
// *****
MPErr MPLibOpen(UInt16 uRefNum)
{
Err err;
GlobalsTypePtr gP = NULL;
	
	// Allocate globals
	ErrFatalDisplayIf(sysInvalidRefNum == uRefNum, "Invalid refnum.");
	
	gP = LockGlobals(uRefNum);
	
	// If this returns NULL, that means we need to allocate the globals.  This also
	// implies that this is the first time we've opened this shared library.  
	if ( !gP ) {
		gP = AllocGlobals( uRefNum );
		if ( !gP ) return MPErrNoGlobals;
			
		// Initialize globals here:
		gP->iOpenCount = 1;

		// Dump diagnostic info, i.e. "MSL ref# %d initially opened; globals initialized.\n", uRefNum
	}
	else
		gP->iOpenCount++;
		
	err = UnlockGlobals( gP );
	ErrFatalDisplayIf( err, "Unable to unlock lib globals.");			
	
	return MPErrNone;
}


// *****
// * FUNCTION: 		Close
// * 
// * DESCRIPTION:	Close MSL; free globals if necessary
// *
// * PARAMETERS:	uRefNum			-	Lib refnum 
// *		      		dwRefCountP	- (Modified) UInt32 into which we put the open count
// *			
// * RETURNED:		MPErrNone		-	success
// *		      		MPErrNoGlobals - Unable to lock down the globals, this is bad
// *
// * POSTCONDITION:	Caller should ALWAYS check dwRefCount upon successful return.  If it's
// *				zero, caller should SysLibRemove() this library as it's no longer in use.
// *
// *	JeffI	05/19/99	Initial Implementation
// *****
MPErr MPLibClose(UInt16 uRefNum, UInt32* dwRefCountP)
{
Err err;
GlobalsTypePtr gP = NULL;
	
	ErrFatalDisplayIf(sysInvalidRefNum == uRefNum, "Invalid refnum.");
	
	if ( !dwRefCountP )														// Validate param
		return MPErrParam;
		
	gP = LockGlobals ( uRefNum );
	if ( !gP ) return MPErrNoGlobals;
	
	gP->iOpenCount--;
	ErrNonFatalDisplayIf( gP->iOpenCount < 0, "Library globals underlock." );

	*dwRefCountP = gP->iOpenCount;
		
	UnlockGlobals( gP );

	if ( *dwRefCountP <= 0 )
	{ // Use this instead of gP->iOpenCount, since we just unlocked gp!
		// Dump diagnostic info i.e.  "MSL ref# %d closed; globals freed.", uRefNum		
		err = FreeGlobals( uRefNum );								
		ErrFatalDisplayIf( err, "Unable to free lib globals.");
	}
	
	return MPErrNone;
}


// *****
// * FUNCTION: 		Sleep
// * 
// * DESCRIPTION:	Called when device goes to sleep.  Since this routine can sometimes be
// *				called from an interrupt handler, you can never spend a lot of time in 
// *				this routine or else you'll make the system unstable and probably cause
// *				mysterious crashes.  In addition, this routine is called as a result of
// *				a battery pull situation; in that case, the Palm is running off of its
// *				super cap, which means there's about 1 ms of processor time remaining
// *				before there is no power.  To avoid catastrophic failure, you and any
// *				other installed shared libraries had better not take up too many cycles
// *				in their respective Sleep function!!
// *
// * PARAMETERS:	uRefNum		-		Lib refnum
// *			
// * RETURNS:		Always 0.  I don't know who uses this return value, or if it's needed.
// *
// *	JeffI	05/19/99	Initial Implementation
// *****
Err MPLibSleep(UInt16 uRefNum)
{
	// If you were implementing custom hardware, you'd do something like
	// this to put it to sleep to conserve power, and to prevent it from
	// sapping the super cap in the event of a battery pull:

	#ifdef MY_HARDWARE_INSTALLED	// ... fictitious example #define ...
		// Tell MyHardware to power down, and then return ASAP!
		MyHardwareBaseAddr->pwrCtlReg |= SLEEP_MODE;
	#endif
	
	return 0;
}


// *****
// * FUNCTION: 		Wake
// * 
// * DESCRIPTION:	Called when device wakes up from sleep.  Since this routine is sometimes
// *				called from an interrupt handler, you can never spend a lot of time in 
// *				this routine or else you'll make the system unstable and probably cause
// *				mysterious crashes.  If you have a time-consuming chore to do, consider
// *				using an interrupt-safe routine like EvtEnqueueKey() to set a flag.  In
// *				an EvtGetEvent hook, you can see this flag come through.  Since you're no
// *				longer in an intrreupt handler, you can do your time-consuming chore at 
// *				that time.
// *
// * PARAMETERS:	uRefNum		-		Lib refnum
// *
// * RETURNS:		Always 0.  I don't know why uses this return value, or if it's needed.
// *			
// *	JeffI	05/19/99	Initial Implementation
// *****
Err MPLibWake(UInt16 uRefNum)
{
	// If you were implementing custom hardware, you'd do something like
	// this to wake your hardware back up:

	#ifdef MY_HARDWARE_INSTALLED	// ... fictitious example #define ...
		// Tell MyHardware to wake up from sleep mode
		MyHardwareBaseAddr->pwrCtlReg &= ~SLEEP_MODE;
	#endif
	
	return 0;
}


// Here are those functions. Note it's a long list!
INT *MPLibMP_new(UInt16 uRefNum)
{
	return MP_new();
}

void	MPLibMP_clear(UInt16 uRefNum, INT *integer)
{
	MP_clear(integer);
}

void MPLibMP_free(UInt16 uRefNum, INT *integer)
{
	MP_free(integer);
}

void MPLibMP_clear_free(UInt16 uRefNum, INT *integer)
{
	MP_clear_free(integer);
}

MP_CTX *MPLibMP_CTX_new(UInt16 uRefNum)
{
	return MP_CTX_new();
}

void MPLibMP_CTX_free(UInt16 uRefNum, MP_CTX *c)
{
	MP_CTX_free(c);
}

MP_MONT_CTX *MPLibMP_MONT_CTX_new(UInt16 uRefNum)
{
	return MP_MONT_CTX_new();
}

void MPLibMP_MONT_CTX_free(UInt16 uRefNum, MP_MONT_CTX *mont)
{
	MP_MONT_CTX_free(mont);
}

Int16 MPLibMP_MONT_CTX_set(UInt16 uRefNum, MP_MONT_CTX *mont, INT *modulus,
    MP_CTX *ctx)
{
	return MP_MONT_CTX_set(mont, modulus, ctx);
}

Int16 MPLibMP_mask_bits(UInt16 uRefNum, INT *a, Int16 n)
{
	return MP_mask_bits(a, n);
}

INT *MPLibMP_copy(UInt16 uRefNum, INT *dest, INT *src)
{
	return MP_copy(dest, src);
}

Int16	MPLibMP_add(UInt16 uRefNum, INT *r, INT *a, INT *b)
{
	return MP_add(r, a, b);
}

void MPLibMP_qadd(UInt16 uRefNum, INT *r, INT *a, INT *b)
{
	MP_qadd(r, a, b);
}

Int16	MPLibMP_sub(UInt16 uRefNum, INT *r, INT *a, INT *b)
{
	return MP_sub(r, a, b);
}

void MPLibMP_qsub(UInt16 uRefNum, INT *r, INT *a, INT *b)
{
	MP_qsub(r, a, b);
}

Int16 MPLibMP_mul(UInt16 uRefNum, INT *r, INT *a, INT *b)
{
	return MP_mul(r, a, b);
}

DIGIT MPLibMP_mul_digit(UInt16 uRefNum, DIGIT *rp, DIGIT *ap, Int16 num, DIGIT w)
{
  return MP_mul_digit(rp, ap, num, w);
}

DIGIT MPLibMP_mul_add_digit(UInt16 uRefNum, DIGIT *rp, DIGIT *ap, Int16 num, DIGIT w)
{
  return MP_mul_add_digit(rp, ap, num, w);
}

Int16 MPLibMP_sqr(UInt16 uRefNum, INT *r, INT *a)
{
	return MP_sqr(r, a);
}

Int16 MPLibMP_div(UInt16 uRefNum, INT *q, INT *r, INT *dividend, INT *divisor, MP_CTX *ctx)
{
	return MP_div(q, r, dividend, divisor, ctx);
}

Int16 MPLibMP_lshift(UInt16 uRefNum, INT *to, INT *from, Int16 n)
{
	return MP_lshift(to, from, n);
}

Int16 MPLibMP_lshift1(UInt16 uRefNum, INT *r, INT *a)
{
	return MP_lshift1(r, a);
}

Int16 MPLibMP_rshift(UInt16 uRefNum, INT *to, INT *from, Int16 n)
{
	return MP_rshift(to, from, n);
}

Int16 MPLibMP_rshift1(UInt16 uRefNum, INT *r, INT *a)
{
	return MP_rshift1(r, a);
}

Int16 MPLibMP_cmp(UInt16 uRefNum, INT *a, INT *b)
{
	return MP_cmp(a, b);
}

Int16 MPLibMP_ucmp(UInt16 uRefNum, INT *a, INT *b)
{
	return MP_ucmp(a, b);
}

Int16 MPLibMP_is_bit_set(UInt16 uRefNum, INT *a, Int16 n)
{
	return MP_is_bit_set(a, n);
}

Int16 MPLibMP_mod_mul(UInt16 uRefNum, INT *r, INT *a, INT *b, INT *m, MP_CTX *ctx)
{
	return MP_mod_mul(r, a, b, m, ctx);
}

Int16 MPLibMP_mont_mul(UInt16 uRefNum, INT *r, INT *a, INT *b,
    MP_MONT_CTX *mont, MP_CTX *ctx)
{
  return MP_mont_mul(r, a, b, mont, ctx);
}

INT *MPLibMP_mod_inverse(UInt16 uRefNum, INT *a, INT *n, MP_CTX *ctx)
{
	return MP_mod_inverse(a, n, ctx);
}

Int16 MPLibMP_mod_exp(UInt16 uRefNum, INT *r, INT *a, INT *b, INT *m, MP_CTX *ctx)
{
	return MP_mod_exp(r, a, b, m, ctx);
}

Int16 MPLibMP_mont_exp(UInt16 uRefNum, INT *r, INT *a, INT *e, MP_MONT_CTX *mont, MP_CTX *ctx)
{
	return MP_mont_exp(r, a, e, mont, ctx);
}

Int16 MPLibMP_gcd(UInt16 uRefNum, INT *d, INT *a, INT *b, MP_CTX *ctx)
{
	return MP_gcd(d, a, b, ctx);
}

Int16 MPLibMP_binary_gcd(UInt16 uRefNum, INT *d, INT *a, INT *b, MP_CTX *ctx)
{
	return MP_binary_gcd(d, a, b, ctx);
}

Int16 MPLibMP_crt2(UInt16 uRefNum, INT *x, INT *v1, INT *v2, INT *p, INT *q, MP_CTX *ctx)
{
	return MP_crt2(x, v1, v2, p, q, ctx);
}

Int16 MPLibMP_probab_prime(UInt16 uRefNum, INT *n, UInt16 reps, MP_CTX *ctx)
{
	return MP_probab_prime(n, reps, ctx);
}

INT *MPLibMP_bin2bn(UInt16 uRefNum, UInt8 *s, UInt16 len)
{
	return MP_bin2bn(s, len);
}

UInt16 MPLibMP_bn2bin(UInt16 uRefNum, INT *a, UInt8 *to)
{
	return MP_bn2bin(a, to);
}

INT *MPLibMP_ascii2bn(UInt16 uRefNum, Char *str)
{
	return MP_ascii2bn(str);
}

Char *MPLibMP_bn2ascii(UInt16 uRefNum, INT *a)
{
	return MP_bn2ascii(a);
}

UInt32 MPLibMP_num_bits(UInt16 uRefNum, INT *a)
{
	return MP_num_bits(a);
}

UInt16 MPLibMP_num_bits_digit(UInt16 uRefNum, DIGIT l)
{
  return MP_num_bits_digit(l);
}
